using System;
using System.Xml;

using gov.va.med.vbecs.DAL.VistALink.OpenLibrary;

namespace gov.va.med.vbecs.DAL.VistALink.OpenLibrary.Messages
{
	/// <summary>
	/// Select division security request message defined in VistALink protocol. 
	/// </summary>
	public class SecuritySelectDivisionRequestMessage : SecurityBaseRequestMessage
	{
		private string _divisionIen; // Internal entry number of division 

		/// <summary>
		/// VistALink message request type string. 
		/// This constant is made public to recognize message XML.
		/// </summary>
		public const string RequestType = "AV.SelectDivision";

		/// <summary>
		/// VistALink message XSD schema file name string. 		
		/// This constant is made public to recognize message XML.
		/// </summary>
		public const string XsdName = "secDivisionRequest.xsd";

		// Constants used in XML message serialization/deserialization
		private const string XMLCONSTS_DIVISION_NODE_NAME = "Division";
		private const string XMLCONSTS_IEN_DIVISION_ATTRIBUTE_NAME = "ien";

		/// <summary>
		/// Select division security message constructor. 
		/// </summary>
		/// <param name="divisionIen">Internal entry number of division to select. Required.</param>
		public SecuritySelectDivisionRequestMessage( string divisionIen )
			: base( XsdName, RequestType )
		{
			if( divisionIen == null )
				throw( new ArgumentNullException( "divisionIen" ) );
			
			if( !IsValidDivisionIen( divisionIen ) )
				throw( new ArgumentOutOfRangeException( "divisionIen", SR.Exceptions.SecDivisionRequestMessageInvalidIEN( divisionIen ) ) );

			_divisionIen = divisionIen;
		}

		/// <summary>
		/// Standard VistALink message deserialization constructor. 
		/// Allows recreating message object from XML document.
		/// </summary>
		/// <param name="sourceDoc">Source XML document.</param>
		public SecuritySelectDivisionRequestMessage( XmlDocument sourceDoc )
			: base( XsdName, RequestType, sourceDoc )
		{
			XmlElement _divisionElement = XmlUtility.ParseGetRequiredElementByUniqueTagName( XmlParseGetRequestElement( sourceDoc ), XMLCONSTS_DIVISION_NODE_NAME );

			_divisionIen = XmlUtility.ParseGetRequiredAttributeValue( _divisionElement, XMLCONSTS_IEN_DIVISION_ATTRIBUTE_NAME );

			if( !IsValidDivisionIen( _divisionIen ) )
				throw( new XmlParseException( SR.Exceptions.SecDivisionRequestMessageInvalidIEN( _divisionIen ) ) );
		}

		/// <summary>
		/// Method checks if specified division IEN is valid. 
		/// </summary>
		/// <param name="testDivisionIen">Internal entry number of division.</param>
		/// <returns>True if IEN is valid. Otherwise - false.</returns>
		private bool IsValidDivisionIen( string testDivisionIen )
		{
			return testDivisionIen.Trim() != String.Empty;
		}

		/// <summary>
		/// XML serialization method adding internal entry number of division to 
		/// request element created by the base class.
		/// </summary>
		/// <param name="writer">XmlWriter to use.</param>
		protected override void WriteXmlRequestNodeContent( XmlWriter writer )
		{
			if( writer == null )
				throw( new ArgumentNullException( "writer" ) );

			writer.WriteStartElement( XMLCONSTS_DIVISION_NODE_NAME );
			writer.WriteAttributeString( XMLCONSTS_IEN_DIVISION_ATTRIBUTE_NAME, _divisionIen );
			writer.WriteEndElement();
		}

		/// <summary>
		/// Internal entry number of division to select. 
		/// </summary>
		public string DivisionIen 
		{
			get
			{
				return _divisionIen;
			}
		}
	}
}
